/************************************************************************
* FILE          : SDS_tclXmlParser.cpp                                 *
* SW-COMPONENT  : SDS_helpclasses                                      *
* DESCRIPTION   : This is the xml parser ported to OSAL compliance     *
*                 dynamic context handling                             *
* COPYRIGHT     : (c) 2011 RBEI, Bangalore                             *
*                                                                      *
* HISTORY:                                                             *
*                                                                      *
*  DD.MM.YYYY | Version | Author                    | Modification     *
*  --------------------------------------------------------------------*
*  31.01.2011 |   1.0   | RBEI/ECF1 - Jose Antony   | Ported to OSAL   *
************************************************************************/


/**
****************************************************************************
* <P> XML.c - implementation file for basic XML parser written in ANSI C++
* for portability. It works by using recursion and a node tree for breaking
* down the elements of an XML document.  </P>
*
* @version     V2.23
* @author      Frank Vanden Berghen
*
*
* BSD license:
* Copyright (c) 2002, Frank Vanden Berghen
* All rights reserved.
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*     * Neither the name of the Frank Vanden Berghen nor the
*       names of its contributors may be used to endorse or promote products
*       derived from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
****************************************************************************
*/


//#ifndef _CRT_SECURE_NO_DEPRECATE
//   #define _CRT_SECURE_NO_DEPRECATE
//#endif

#include "sds_AllHelpclasses.h"

#ifndef _T
   #define _T(c) c
#endif


static inline tS32 OSAL_MyStricmp(tCString s1, tCString s2)
{
   tS32 c1, c2;
   for(;;)
   {
      c1 = OSAL_s32ToLower( (tU8) *s1++ );
      c2 = OSAL_s32ToLower( (tU8) *s2++ );
      if (c1 == 0 || c1 != c2)
         return c1 - c2;
   }
}
static inline tVoid OSAL_MyFree(tPVoid pvMemoryToFree)
{
   if(pvMemoryToFree)
   {
      tPU32 pMem = ((tPU32) pvMemoryToFree) - 1;
      OSAL_vMemoryFree(pMem);
      pvMemoryToFree = OSAL_NULL;
   }
}

static inline tPVoid OSAL_MyMalloc(tSize uNewSize)
{
   tPVoid pvAllocatedMemory = OSAL_NULL;
   if(uNewSize>0)
   {
      pvAllocatedMemory = OSAL_pvMemoryAllocate( uNewSize+sizeof(tU32) );
      if(OSAL_NULL != pvAllocatedMemory)
      {
         *((tPU32) pvAllocatedMemory) = uNewSize;
         return (((tPU8)pvAllocatedMemory)+sizeof(tU32));
      }
   }
   return pvAllocatedMemory;
}

static inline tPVoid OSAL_MyRealloc(tPVoid pvMemoryToRealloc,tSize uNewSize)
{
   tPVoid pvAllocatedMemory = OSAL_NULL;

   if (pvMemoryToRealloc == OSAL_NULL) //If null,. act as malloc()
   {
      if(uNewSize>0)
      {
         pvAllocatedMemory = OSAL_MyMalloc(uNewSize);
      }        
   }
   else
   {
      if(uNewSize == 0) //If size is zero, act as free()
      {
         OSAL_MyFree(pvMemoryToRealloc);            
      }
      else
      {
         pvAllocatedMemory = OSAL_MyMalloc(uNewSize);
         if(OSAL_NULL != pvAllocatedMemory)
         {
            tSize uOldSize = *(((tPU32)pvMemoryToRealloc)-1);
            OSAL_pvMemoryCopy(pvAllocatedMemory, pvMemoryToRealloc, (uOldSize < uNewSize) ? uOldSize : uNewSize);
            //Free the initial memory
            OSAL_MyFree(pvMemoryToRealloc);
         }
      }
   }
   return pvAllocatedMemory;
}

XMLCSTR XMLNode::getVersion() { return _T("v2.23"); }
tVoid free_XMLDLL(tPVoid t){OSAL_MyFree(t);}

static tChar strictUTF8Parsing=1, guessUnicodeChars=1, dropWhiteSpace=1;

inline tS32 mmin( tCS32 t1, tCS32 t2 ) { return t1 < t2 ? t1 : t2; }


// You can modify the initialization of the variable "XMLClearTags" below
// to change the clearTags that are currently recognized by the library.
// The number on the second columns is the length of the string inside the
// first column. The "<!DOCTYPE" declaration must be the second in the list.
static ALLXMLClearTag XMLClearTags[] =
{
   {    _T("<![CDATA["),9,  _T("]]>")      },
   {    _T("<!DOCTYPE"),9,  _T(">")        },
   {    _T("<PRE>")    ,5,  _T("</PRE>")   },
   {    _T("<Script>") ,8,  _T("</Script>")},
   {    _T("<!--")     ,4,  _T("-->")      },
   {    OSAL_NULL           ,0,  OSAL_NULL           }
};
ALLXMLClearTag* XMLNode::getClearTagTable() { return XMLClearTags; }

// You can modify the initialization of the variable "XMLEntities" below
// to change the character entities that are currently recognized by the library.
// The number on the second columns is the length of the string inside the
// first column. Additionally, the syntaxes "&#xA0;" and "&#160;" are recognized.
typedef struct { XMLCSTR s; tS32 l; XMLCHAR c;} XMLCharacterEntity;
static XMLCharacterEntity XMLEntities[] =
{
   { _T("&amp;" ), 5, _T('&' )},
   { _T("&lt;"  ), 4, _T('<' )},
   { _T("&gt;"  ), 4, _T('>' )},
   { _T("&quot;"), 6, _T('\"')},
   { _T("&apos;"), 6, _T('\'')},
   { NULL        , 0, '\0'    }
};

// When rendering the XMLNode to a string (using the "createXMLString" function),
// you can ask for a beautiful formatting. This formatting is using the
// following indentation character:
#define INDENTCHAR _T('\t')

// The following function parses the XML errors into a user friendly string.
// You can edit this to change the output language of the library to something else.
XMLCSTR XMLNode::getError(XMLError xerror)
{
   switch (xerror)
   {
   case eXMLErrorNone:                  return _T("No error");
   case eXMLErrorMissingEndTag:         return _T("Warning: Unmatched end tag");
   case eXMLErrorEmpty:                 return _T("Error: No XML data");
   case eXMLErrorFirstNotStartTag:      return _T("Error: First token not start tag");
   case eXMLErrorMissingTagName:        return _T("Error: Missing start tag name");
   case eXMLErrorMissingEndTagName:     return _T("Error: Missing end tag name");
   case eXMLErrorNoMatchingQuote:       return _T("Error: Unmatched quote");
   case eXMLErrorUnmatchedEndTag:       return _T("Error: Unmatched end tag");
   case eXMLErrorUnmatchedEndClearTag:  return _T("Error: Unmatched clear tag end");
   case eXMLErrorUnexpectedToken:       return _T("Error: Unexpected token found");
   case eXMLErrorInvalidTag:            return _T("Error: Invalid tag found");
   case eXMLErrorNoElements:            return _T("Error: No elements found");
   case eXMLErrorFileNotFound:          return _T("Error: File not found");
   case eXMLErrorFirstTagNotFound:      return _T("Error: First Tag not found");
   case eXMLErrorUnknownCharacterEntity:return _T("Error: Unknown character entity");
   case eXMLErrorCharConversionError:   return _T("Error: unable to convert between UNICODE and MultiByte chars");
   case eXMLErrorCannotOpenWriteFile:   return _T("Error: unable to open file for writing");
   case eXMLErrorCannotWriteFile:       return _T("Error: cannot write into file");
   };
   return _T("Unknown");
}


/////////////////////////////////////////////////////////////////////////
//      Here start the core implementation of the XMLParser library    //
/////////////////////////////////////////////////////////////////////////

// You should normally not change anything below this point.
// For your own information, I suggest that you read the openFileHelper below:
XMLNode XMLNode::openFileHelper(XMLCSTR filename, XMLCSTR tag)
{
   // guess the value of the global parameter "strictUTF8Parsing"
   // (the guess is based on the first 200 bytes of the file).
   /*  FILE *f=_tfopen(filename,_T("rb"));
   if (f)
   {
   char bb[205];
   tS32 l=(tS32)fread(bb,1,200,f);
   setGlobalOptions(guessUnicodeChars,guessUTF8ParsingParameterValue(bb,l),dropWhiteSpace);
   fclose(f);
   }
   */
   // parse the file
   XMLResults pResults;
   XMLNode xnode=XMLNode::parseFile(filename,tag,&pResults);

   // display error message (if any)
   if (pResults.error != eXMLErrorNone)
   {
      // create message
      //char message[2000],*s1=(tString)"",*s3=(tString)""; XMLCSTR s2=_T("");
      //if (pResults.error==eXMLErrorFirstTagNotFound) { s1=(tString)"First Tag should be '"; s2=tag; s3=(tString)"'.\n"; }
      //sprintf(message,
      //   ,filename,XMLNode::getError(pResults.error),pResults.nLine,pResults.nColumn,s1,s2,s3);
      //printf("%s",message);


   }
   return xnode;
}



static tCC8 XML_utf8ByteTable[256] =
{
   //  0 1 2 3 4 5 6 7 8 9 a b c d e f
   0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x00
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x10
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x20
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x30
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x40
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x50
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x60
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x70End of ASCII range
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x80 0x80 to 0xc1 invalid
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0x90
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0xa0
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,// 0xb0
   1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,// 0xc0 0xc2 to 0xdf 2 byte
   2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,// 0xd0
   3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,// 0xe0 0xe0 to 0xef 3 byte
   4,4,4,4,4,1,1,1,1,1,1,1,1,1,1,1 // 0xf0 0xf0 to 0xf4 4 byte, 0xf5 and higher invalid
};
static tCC8 XML_asciiByteTable[256] =
{
   0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1
};
static tCString XML_ByteTable=(tCString )XML_utf8ByteTable; // the default is "strictUTF8Parsing=1";


XMLError XMLNode::writeToFile(XMLCSTR filename, tCString encoding, tChar nFormat) const
{
   tS32        i;
   tS32        s32Result = OSAL_OK;
   SDS_tclFile oNewFile;
   XMLSTR      t = createXMLString(nFormat,&i);

   if(TRUE == oNewFile.bIsFileAvailable(filename))
   {
      if(OSAL_OK!= OSAL_s32IORemove(filename))
      {
         OSAL_MyFree(t);
         return eXMLErrorCannotWriteFile;
      }
   }

   if(FALSE == oNewFile.bCreate(filename,OSAL_EN_READWRITE))
   {
      OSAL_MyFree(t);
      return eXMLErrorCannotWriteFile;
   }
   if (!isDeclarationSet())
   {
      if ((!encoding)||(XML_ByteTable==XML_utf8ByteTable))
      {
         // header so that windows recognize the file as UTF-8:
         tU8 h[3]={0xEF,0xBB,0xBF};

         s32Result = oNewFile.s32Write((tString)h,3);
         if(OSAL_ERROR == s32Result)
         {
            oNewFile.vClose();
            (tVoid)OSAL_s32IORemove(filename);
            OSAL_MyFree(t);
            return eXMLErrorCannotWriteFile;
         }
         s32Result = oNewFile.s32Write((tString)"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n",39);
         if(OSAL_ERROR == s32Result)
         {
            oNewFile.vClose();
            (tVoid)OSAL_s32IORemove(filename);
            OSAL_MyFree(t);
            return eXMLErrorCannotWriteFile;
         }
      }
      //  else
      //    if (OSALUTIL_s32FPrintf(fd,"<?xml version=\"1.0\" encoding=\"%s\"?>\n",encoding)<0)
      //	{
      //	return eXMLErrorCannotWriteFile;
      //	}
   }
   else
   {
      if (XML_ByteTable==XML_utf8ByteTable) // test if strictUTF8Parsing==1"
      {
         tU8 h[3]={0xEF,0xBB,0xBF};
         s32Result = oNewFile.s32Write((tString)h,3);
         if(OSAL_ERROR == s32Result)
         {
            oNewFile.vClose();
            (tVoid)OSAL_s32IORemove(filename);
            OSAL_MyFree(t);
            return eXMLErrorCannotWriteFile;
         }
      }
   }
   s32Result = oNewFile.s32Write((tString)t,sizeof(XMLCHAR)*i);
   oNewFile.vClose();
   if(OSAL_ERROR == s32Result)
   {
      (tVoid)OSAL_s32IORemove(filename);
      OSAL_MyFree(t);
      return eXMLErrorCannotWriteFile;
   }
   OSAL_MyFree(t);
   return eXMLErrorNone;
}


// Duplicate a given string.
XMLSTR stringDup(XMLCSTR lpszData, tS32 cbData)
{
   if (lpszData==OSAL_NULL) return OSAL_NULL;

   XMLSTR lpszNew;
   if (cbData==0) cbData=(tS32)OSAL_u32StringLength(lpszData);
   lpszNew = (XMLSTR)OSAL_MyMalloc((cbData+1) * sizeof(XMLCHAR));
   if (lpszNew)
   {
      OSAL_pvMemoryCopy(lpszNew, lpszData, (cbData) * sizeof(XMLCHAR));
      lpszNew[cbData] = (XMLCHAR)OSAL_NULL;
   }
   return lpszNew;
}

XMLNode XMLNode::emptyXMLNode;
XMLClear XMLNode::emptyXMLClear={ OSAL_NULL, OSAL_NULL, OSAL_NULL};
XMLAttribute XMLNode::emptyXMLAttribute={ OSAL_NULL, OSAL_NULL};

// Enumeration used to decipher what type a token is
typedef enum
{
   eTokenText = 0,
   eTokenQuotedText,
   eTokenTagStart,         /* "<"            */
   eTokenTagEnd,           /* "</"           */
   eTokenCloseTag,         /* ">"            */
   eTokenEquals,           /* "="            */
   eTokenDeclaration,      /* "<?"           */
   eTokenShortHandClose,   /* "/>"           */
   eTokenClear,
   eTokenError
}XMLTokenTypeTag;

// Main structure used for parsing XML
typedef struct XML
{
   XMLCSTR                lpXML;
   XMLCSTR                lpszText;
   tS32                    nIndex,nIndexMissigEndTag;
   enum XMLError          error;
   XMLCSTR                lpEndTag;
   tS32                    cbEndTag;
   XMLCSTR                lpNewElement;
   tS32                    cbNewElement;
   tS32                    nFirst;
} XML;

typedef struct
{
   ALLXMLClearTag *pClr;
   XMLCSTR     pStr;
} NextToken;

// Enumeration used when parsing attributes
typedef enum
{
   eAttribName = 0,
   eAttribEquals,
   eAttribValue
} Attrib;

// Enumeration used when parsing elements to dictate whether we are currently
// inside a tag
typedef enum
{
   eInsideTag = 0,
   eOutsideTag
} Status;

// private (used while rendering):
XMLSTR toXMLString(XMLSTR dest,XMLCSTR source)
{
   XMLSTR dd=dest;
   tBool bQuit = TRUE;
   XMLCHAR ch;
   if((OSAL_NULL == dest) || (OSAL_NULL == source))
   {
      return OSAL_NULL;
   }
   else
   {
      XMLCharacterEntity *entity;
      ch = *source;
      while(ch != 0)
      {
         entity=XMLEntities;
         do
         {
            if (ch==entity->c)
            {
               OSAL_szStringCopy(dest,entity->s);
               dest+=entity->l;
               source++;
               bQuit = FALSE;
               break;
            }
            entity++;
         } while(entity->s);
         if(TRUE == bQuit)
         {
            switch(XML_ByteTable[(tU8)ch])
            {         
            case 4:
               *(dest++)=*(source++);
               /*fall through*/
            case 3:
               *(dest++)=*(source++);
               /*fall through*/
            case 2:
               *(dest++)=*(source++);
               /*fall through*/
            case 1:
               *(dest++)=*(source++);
               /*fall through*/
            default:
               ;
            }
         }
         ch = *source;
      }
      *dest=0;
      return dd;
   }
}

// private (used while rendering):
tS32 lengthXMLString(XMLCSTR source)
{
   tS32 r=0;
   XMLCharacterEntity *entity;
   tBool bQuit = TRUE;
   XMLCHAR ch;
   ch = *source;
   while(ch != 0)
   {
      entity=XMLEntities;
      do
      {
         if (ch==entity->c)
         {
            r+=entity->l;
            source++;
            bQuit = FALSE;
            break;
         }
         entity++;
      } while(entity->s);

      if(TRUE == bQuit)
      {
      ch=XML_ByteTable[(tU8)ch]; r+=ch; source+=ch;
      }
      ch = *source;
   }
   return r;
}

XMLSTR toXMLString(XMLCSTR source)
{
   XMLSTR dest = OSAL_NULL;
   if(source != OSAL_NULL)
   {
       dest=(XMLSTR)OSAL_MyMalloc((lengthXMLString(source)+1)*sizeof(XMLCHAR));
   }
   if(OSAL_NULL!=dest)
   {
      return toXMLString(dest,source);
   }
   else
   {
      return OSAL_NULL;
   }
}

XMLSTR toXMLStringFast(XMLSTR *dest,tPS32 destSz, XMLCSTR source)
{
   tS32 l=lengthXMLString(source)+1;
   if (l>*destSz) { *destSz=l; *dest=(XMLSTR)OSAL_MyRealloc(*dest,l*sizeof(XMLCHAR)); }
   return toXMLString(*dest,source);
}

// private:
XMLSTR fromXMLString(XMLCSTR s, tS32 lo, XML *pXML)
{
   // This function is the opposite of the function "toXMLString". It decodes the escape
   // sequences &amp;, &quot;, &apos;, &lt;, &gt; and replace them by the characters
   // &,",',<,>. This function is used internally by the XML Parser. All the calls to
   // the XML library will always gives you back "decoded" strings.
   //
   // in: string (s) and length (lo) of string
   // out:  new allocated string converted from xml
   if (!s) return OSAL_NULL;

   tS32 ll=0,j;
   XMLSTR d;
   XMLCSTR ss=s;
   XMLCharacterEntity *entity;
   while ((lo>0)&&(*s))
   {
      if (*s==_T('&'))
      {
         if ((lo>2)&&(s[1]==_T('#')))
         {
            s+=2; lo-=2;
            if ((*s==_T('X'))||(*s==_T('x'))) { s++; lo--; }
            while ((*s)&&(*s!=_T(';'))&&((lo--)>0)) s++;
            if (*s!=_T(';'))
            {
               pXML->error=eXMLErrorUnknownCharacterEntity;
               return OSAL_NULL;
            }
            s++; lo--;
         } else
         {
            entity=XMLEntities;
            do
            {
               if ((lo>=entity->l)&&(OSAL_s32StringNCompare(s,entity->s,entity->l)==0)) { s+=entity->l; lo-=entity->l; break; }
               entity++;
            } while(entity->s);
            if (!entity->s)
            {
               pXML->error=eXMLErrorUnknownCharacterEntity;
               return OSAL_NULL;
            }
         }
      } else
      {
         j=XML_ByteTable[(tU8)*s]; s+=j; lo-=j; ll+=j-1;

      }
      ll++;
   }

   d=(XMLSTR)OSAL_MyMalloc((ll+1)*sizeof(XMLCHAR));
   s=d;
   while (ll-->0)
   {
      if (*ss==_T('&'))
      {
         if (ss[1]==_T('#'))
         {
            ss+=2; j=0;
            if ((*ss==_T('X'))||(*ss==_T('x')))
            {
               ss++;
               while (*ss!=_T(';'))
               {
                  if ((*ss>=_T('0'))&&(*ss<=_T('9'))) j=(j<<4)+*ss-_T('0');
                  else if ((*ss>=_T('A'))&&(*ss<=_T('F'))) j=(j<<4)+*ss-_T('A')+10;
                  else if ((*ss>=_T('a'))&&(*ss<=_T('f'))) j=(j<<4)+*ss-_T('a')+10;
                  else { OSAL_MyFree((tPVoid)s); pXML->error=eXMLErrorUnknownCharacterEntity;return OSAL_NULL;}
                  ss++;
               }
            } else
            {
               while (*ss!=_T(';'))
               {
                  if ((*ss>=_T('0'))&&(*ss<=_T('9'))) j=(j*10)+*ss-_T('0');
                  else { OSAL_MyFree((tPVoid)s); pXML->error=eXMLErrorUnknownCharacterEntity;return OSAL_NULL;}
                  ss++;
               }
            }
            (*d++)=(XMLCHAR)j; ss++;
         } else
         {
            entity=XMLEntities;
            do
            {
               if (OSAL_s32StringNCompare(ss,entity->s,entity->l)==0) { *(d++)=entity->c; ss+=entity->l; break; }
               entity++;
            } while(entity->s);
         }
      } else
      {



         switch(XML_ByteTable[(tU8)*ss])
         {
         case 4:
            *(d++)=*(ss++); ll--;
            /*fall through*/
         case 3:
            *(d++)=*(ss++); ll--;
            /*fall through*/
         case 2:
            *(d++)=*(ss++); ll--;
            /*fall through*/
         case 1:
            *(d++)=*(ss++);
            /*fall through*/
         default:
            ;
         }

      }
   }
   *d=0;
   return (XMLSTR)s;
}

#define XML_isSPACECHAR(ch) ((ch==_T('\n'))||(ch==_T(' '))||(ch== _T('\t'))||(ch==_T('\r')))

// private:
char myTagCompare(XMLCSTR cclose, XMLCSTR copen)
// !!!! WARNING strange convention&:
// return 0 if equals
// return 1 if different
{
   if (!cclose) return 1;
   tS32 l=(tS32)OSAL_u32StringLength(cclose);
   if (OSAL_s32StringNCompare(cclose, copen, l)!=0) return 1;
   const XMLCHAR c=copen[l];
   if (XML_isSPACECHAR(c)||
      (c==_T('/' ))||
      (c==_T('<' ))||
      (c==_T('>' ))||
      (c==_T('=' ))) return 0;
   return 1;
}

// Obtain the next character from the string.
static inline XMLCHAR getNextChar(XML *pXML)
{
   XMLCHAR ch = pXML->lpXML[pXML->nIndex];



   pXML->nIndex+=XML_ByteTable[(tU8)ch];

   return ch;
}

// Find the next token in a string.
// pcbToken contains the number of characters that have been read.
static NextToken GetNextToken(XML *pXML, tPS32 pcbToken, XMLTokenTypeTag *pType)
{
   NextToken        result;
   XMLCHAR            ch;
   XMLCHAR            chTemp;
   tS32              indexStart,nFoundMatch,nIsText=FALSE;
   result.pClr=OSAL_NULL; // prevent warning

   // Find next non-white space character
   do { indexStart=pXML->nIndex; ch=getNextChar(pXML); } while XML_isSPACECHAR(ch);

   if (ch)
   {
      // Cache the current string pointer
      result.pStr = &pXML->lpXML[indexStart];

      // First check whether the token is in the clear tag list (meaning it
      // does not need formatting).
      ALLXMLClearTag *ctag=XMLClearTags;
      do
      {
         if (OSAL_s32StringNCompare(ctag->lpszOpen, result.pStr, ctag->openTagLen)==0)
         {
            result.pClr=ctag;
            pXML->nIndex+=ctag->openTagLen-1;
            *pType=eTokenClear;
            return result;
         }
         ctag++;
      } while(ctag->lpszOpen);

      // If we didn't find a clear tag then check for standard tokens
      switch(ch)
      {
         // Check for quotes
      case _T('\''):
      case _T('\"'):
         // Type of token
         *pType = eTokenQuotedText;
         chTemp = ch;

         // Set the size
         nFoundMatch = FALSE;

         // Search through the string to find a matching quote
         ch = getNextChar(pXML);
         while(ch != 0)
         {
            if (ch==chTemp) 
            { 
               nFoundMatch = TRUE; 
               break; 
            }
            if (ch==_T('<')) 
            {
               break;
            }
            ch = getNextChar(pXML);
         }

         // If we failed to find a matching quote
         if (nFoundMatch == FALSE)
         {
            pXML->nIndex=indexStart+1;
            nIsText=TRUE;
            break;
         }

         //  4.02.2002
         //            if (FindNonWhiteSpace(pXML)) pXML->nIndex--;

         break;

         // Equals (used with attribute values)
      case _T('='):
         *pType = eTokenEquals;
         break;

         // Close tag
      case _T('>'):
         *pType = eTokenCloseTag;
         break;

         // Check for tag start and tag end
      case _T('<'):

         // Peek at the next character to see if we have an end tag '</',
         // or an xml declaration '<?'
         chTemp = pXML->lpXML[pXML->nIndex];

         // If we have a tag end...
         if (chTemp == _T('/'))
         {
            // Set the type and ensure we point at the next character
            getNextChar(pXML);
            *pType = eTokenTagEnd;
         }

         // If we have an XML declaration tag
         else if (chTemp == _T('?'))
         {

            // Set the type and ensure we point at the next character
            getNextChar(pXML);
            *pType = eTokenDeclaration;
         }

         // Otherwise we must have a start tag
         else
         {
            *pType = eTokenTagStart;
         }
         break;

         // Check to see if we have a short hand type end tag ('/>').
      case _T('/'):

         // Peek at the next character to see if we have a short end tag '/>'
         chTemp = pXML->lpXML[pXML->nIndex];

         // If we have a short hand end tag...
         if (chTemp == _T('>'))
         {
            // Set the type and ensure we point at the next character
            getNextChar(pXML);
            *pType = eTokenShortHandClose;
            break;
         }

         // If we haven't found a short hand closing tag then drop into the
         // text process
         nIsText = TRUE;
         break;

         // Other characters
      default:
         nIsText = TRUE;
      }

      // If this is a TEXT node
      if (nIsText)
      {
         // Indicate we are dealing with text
         *pType = eTokenText;
         ch = getNextChar(pXML);
         while(ch != 0)
         {
            if XML_isSPACECHAR(ch)
            {
               indexStart++; 
               break;
            } 
            else if(ch==_T('/'))
            {
               // If we find a slash then this maybe text or a short hand end tag
               // Peek at the next character to see it we have short hand end tag
               ch=pXML->lpXML[pXML->nIndex];
               // If we found a short hand end tag then we need to exit the loop
               if (ch==_T('>')) 
               { 
                  pXML->nIndex--; 
                  break; 
               }
            } 
            else if((ch==_T('<')) || (ch==_T('>')) || (ch==_T('=')))
            {
               pXML->nIndex--; 
               break;
            }
            ch = getNextChar(pXML);
         }
      }
      *pcbToken = pXML->nIndex-indexStart;
   } else
   {
      // If we failed to obtain a valid character
      *pcbToken = 0;
      *pType = eTokenError;
      result.pStr=OSAL_NULL;
   }

   return result;
}


XMLCSTR XMLNode::updateName_WOSD(XMLCSTR lpszName)
{
   if(d->lpszName && (lpszName != d->lpszName)) 
   {
      OSAL_MyFree((tPVoid)d->lpszName);
   }
   d->lpszName = lpszName;
   return lpszName;
}


// private:
XMLNode::XMLNode(struct XMLNodeDataTag *p){ d=p; (p->ref_count)++; }
XMLNode::XMLNode(XMLNodeData *pParent, XMLCSTR lpszName, tChar isDeclaration)
{
   d=(XMLNodeData*)OSAL_MyMalloc(sizeof(XMLNodeData));
   d->ref_count=1;

   d->lpszName=OSAL_NULL;
   d->nChild= 0;
   d->nText = 0;
   d->nClear = 0;
   d->nAttribute = 0;

   d->isDeclaration = isDeclaration;

   d->pParent = pParent;
   d->pChild= OSAL_NULL;
   d->pText= OSAL_NULL;
   d->pClear= OSAL_NULL;
   d->pAttribute= OSAL_NULL;
   d->pOrder= OSAL_NULL;

   updateName_WOSD(lpszName);
}

XMLNode XMLNode::createXMLTopNode_WOSD(XMLCSTR lpszName, tChar isDeclaration) { return XMLNode(OSAL_NULL,lpszName,isDeclaration); }
XMLNode XMLNode::createXMLTopNode(XMLCSTR lpszName, tChar isDeclaration) { return XMLNode(OSAL_NULL,stringDup(lpszName),isDeclaration); }


#define MEMORYINCREASE 50

static inline tPVoid myRealloc(tPVoid p, tS32 newsize, tS32 memInc, tS32 sizeofElem)
{
   if (p==OSAL_NULL) 
   {
      if (memInc)
         return OSAL_MyMalloc(memInc*sizeofElem);
      return OSAL_MyMalloc(sizeofElem);
   }
   if ((memInc==0)||((newsize%memInc)==0))
      p=OSAL_MyRealloc(p,(newsize+memInc)*sizeofElem);
   //    if (!p)
   //    {
   //        printf("XMLParser Error: Not enough memory! Aborting...\n"); exit(220);
   //    }
   return p;
}

// private:
tS32 XMLNode::findPosition(XMLNodeData *dq, tS32 indexe, XMLElementType xtype)
{
   if (indexe<0) return -1;
   tS32 i=0,j=(tS32)((indexe<<2)+xtype),*o=dq->pOrder; while (o[i]!=j) i++; return i;
}

// private:
// update "order" information when deleting a content of a XMLNode
tS32 XMLNode::removeOrderElement(XMLNodeData *dq, XMLElementType t, tS32 indexe)
{
   tS32 n=dq->nChild+dq->nText+dq->nClear, *o=dq->pOrder,i=findPosition(dq,indexe,t);
   OSAL_pvMemoryMove(o+i, o+i+1, (n-i)*sizeof(tS32));
   for (;i<n;i++)
      if ((o[i]&3)==(tS32)t) o[i]-=4;
   // We should normally do:
   // d->pOrder=(tS32)realloc(d->pOrder,n*sizeof(tS32));
   // but we skip reallocation because it's too time consuming.
   // Anyway, at the end, it will be free'd completely at once.
   return i;
}

tPVoid XMLNode::addToOrder(tS32 memoryIncrease,tPS32 _pos, tS32 nc, tPVoid p, tS32 size, XMLElementType xtype)
{
   //  in: *_pos is the position inside d->pOrder ("-1" means "EndOf")
   // out: *_pos is the index inside p
   p=myRealloc(p,(nc+1),memoryIncrease,size);
   tS32 n=d->nChild+d->nText+d->nClear;
   d->pOrder=(tPS32)myRealloc(d->pOrder,n+1,memoryIncrease*3,sizeof(tS32));
   tS32 pos=*_pos,*o=d->pOrder;

   if ((pos<0)||(pos>=n)) { *_pos=nc; o[n]=(tS32)((nc<<2)+xtype); return p; }

   tS32 i=pos;
   OSAL_pvMemoryMove(o+i+1, o+i, (n-i)*sizeof(tS32));

   while ((pos<n)&&((o[pos]&3)!=(tS32)xtype)) pos++;
   if (pos==n) { *_pos=nc; o[n]=(tS32)((nc<<2)+xtype); return p; }

   o[i]=o[pos];
   for (i=pos+1;i<=n;i++) if ((o[i]&3)==(tS32)xtype) o[i]+=4;

   *_pos=pos=o[pos]>>2;
   OSAL_pvMemoryMove(((tString)p)+(pos+1)*size,((tString)p)+pos*size,(nc-pos)*size);

   return p;
}

// Add a child node to the given element.
XMLNode XMLNode::addChild_priv(tS32 memoryIncrease, XMLCSTR lpszName, tChar isDeclaration, tS32 pos)
{
   if (!lpszName) return emptyXMLNode;
   d->pChild=(XMLNode*)addToOrder(memoryIncrease,&pos,d->nChild,d->pChild,sizeof(XMLNode),eNodeChild);
   d->pChild[pos].d=OSAL_NULL;
   d->pChild[pos]=XMLNode(d,lpszName,isDeclaration);
   d->nChild++;
   return d->pChild[pos];
}

// Add an attribute to an element.
XMLAttribute *XMLNode::addAttribute_priv(tS32 memoryIncrease,XMLCSTR lpszName, XMLCSTR lpszValuev)
{
   if (!lpszName) return &emptyXMLAttribute;
   tS32 nc=d->nAttribute;
   d->pAttribute=(XMLAttribute*)myRealloc(d->pAttribute,(nc+1),memoryIncrease,sizeof(XMLAttribute));
   XMLAttribute *pAttr=d->pAttribute+nc;
   pAttr->lpszName = lpszName;
   pAttr->lpszValue = lpszValuev;
   d->nAttribute++;
   return pAttr;
}

// Add text to the element.
XMLCSTR XMLNode::addText_priv(tS32 memoryIncrease, XMLCSTR lpszValue, tS32 pos)
{
   if (!lpszValue) return OSAL_NULL;
   d->pText=(XMLCSTR*)addToOrder(memoryIncrease,&pos,d->nText,d->pText,sizeof(XMLSTR),eNodeText);
   d->pText[pos]=lpszValue;
   d->nText++;
   return lpszValue;
}

// Add clear (unformatted) text to the element.
XMLClear *XMLNode::addClear_priv(tS32 memoryIncrease, XMLCSTR lpszValue, XMLCSTR lpszOpen, XMLCSTR lpszClose, tS32 pos)
{
   if (!lpszValue) return &emptyXMLClear;
   d->pClear=(XMLClear *)addToOrder(memoryIncrease,&pos,d->nClear,d->pClear,sizeof(XMLClear),eNodeClear);
   XMLClear *pNewClear=d->pClear+pos;
   pNewClear->lpszValue = lpszValue;
   if (!lpszOpen) lpszOpen=getClearTagTable()->lpszOpen;
   if (!lpszClose) lpszOpen=getClearTagTable()->lpszClose;
   pNewClear->lpszOpenTag = lpszOpen;
   pNewClear->lpszCloseTag = lpszClose;
   d->nClear++;
   return pNewClear;
}

// private:
// Parse a clear (unformatted) type node.
tChar XMLNode::parseClearTag(tPVoid px, ALLXMLClearTag *pClear)
{
   XML *pXML=(XML *)px;
   tS32 cbTemp=0;
   XMLCSTR lpszTemp=OSAL_NULL;
   XMLCSTR lpXML=&pXML->lpXML[pXML->nIndex];
   static XMLCSTR docTypeEnd=_T("]>");

   // Find the closing tag
   // Seems the <!DOCTYPE need a better treatment so lets handle it
   if (pClear->lpszOpen==XMLClearTags[1].lpszOpen)
   {
      XMLCSTR pCh=lpXML;
      while (*pCh)
      {
         if (*pCh==_T('<')) { pClear->lpszClose=docTypeEnd; lpszTemp=(tString)OSAL_ps8StringSubString(lpXML,docTypeEnd); break; }
         else if (*pCh==_T('>')) { lpszTemp=pCh; break; }



         pCh+=XML_ByteTable[(tU8)(*pCh)];

      }
   } else lpszTemp=(tString)OSAL_ps8StringSubString(lpXML, pClear->lpszClose);

   if (lpszTemp)
   {
      // Cache the size and increment the index
      cbTemp = (tS32)(lpszTemp - lpXML);

      pXML->nIndex += cbTemp+(tS32)OSAL_u32StringLength(pClear->lpszClose);

      // Add the clear node to the current element
      addClear_priv(MEMORYINCREASE,stringDup(lpXML,cbTemp), pClear->lpszOpen, pClear->lpszClose,-1);
      return 0;
   }

   // If we failed to find the end tag
   pXML->error = eXMLErrorUnmatchedEndClearTag;
   return 1;
}

tVoid XMLNode::exactMemory(XMLNodeData *dq)
{
   if (dq->pOrder)     dq->pOrder=(tPS32)OSAL_MyRealloc(dq->pOrder,(dq->nChild+dq->nText+dq->nClear)*sizeof(tS32));
   if (dq->pChild)     dq->pChild=(XMLNode*)OSAL_MyRealloc(dq->pChild,dq->nChild*sizeof(XMLNode));
   if (dq->pAttribute) dq->pAttribute=(XMLAttribute*)OSAL_MyRealloc(dq->pAttribute,dq->nAttribute*sizeof(XMLAttribute));
   if (dq->pText)      dq->pText=(XMLCSTR*)OSAL_MyRealloc(dq->pText,dq->nText*sizeof(XMLSTR));
   if (dq->pClear)     dq->pClear=(XMLClear *)OSAL_MyRealloc(dq->pClear,dq->nClear*sizeof(XMLClear));
}

tChar XMLNode::maybeAddTxT(tPVoid pa, XMLCSTR tokenPStr)
{
   XML *pXML=(XML *)pa;
   XMLCSTR lpszText=pXML->lpszText;
   if (!lpszText) return 0;
   while (XML_isSPACECHAR(*lpszText)&&(lpszText!=tokenPStr)) lpszText++;
   tS32 cbText = (tS32)(tokenPStr - lpszText);
   if (!cbText) { pXML->lpszText=OSAL_NULL; return 0; }
   cbText--; while ((cbText)&&XML_isSPACECHAR(lpszText[cbText])) cbText--; cbText++; 
   if (!cbText) { pXML->lpszText=OSAL_NULL; return 0; }
   lpszText=fromXMLString(lpszText,cbText,pXML);
   if (!lpszText) return 1;
   addText_priv(MEMORYINCREASE,lpszText,-1);
   pXML->lpszText=OSAL_NULL;
   return 0;
}
// private:
// Recursively parse an XML element.
tS32 XMLNode::ParseXMLElement(tPVoid pa)
{
   XML *pXML=(XML *)pa;
   tS32 cbToken;
   XMLTokenTypeTag type;
   NextToken token;
   XMLCSTR lpszTemp=OSAL_NULL;
   tS32 cbTemp=0;
   tChar nDeclaration;
   XMLNode pNew;
   Status status; // inside or outside a tag
   Attrib attrib = eAttribName;

   //assert(pXML);

   // If this is the first call to the function
   if (pXML->nFirst)
   {
      // Assume we are outside of a tag definition
      pXML->nFirst = FALSE;
      status = eOutsideTag;
   } else
   {
      // If this is not the first call then we should only be called when inside a tag.
      status = eInsideTag;
   }

   // Iterate through the tokens in the document
   for(;;)
   {
      // Obtain the next token
      token = GetNextToken(pXML, &cbToken, &type);

      if (type != eTokenError)
      {
         // Check the current status
         switch(status)
         {

            // If we are outside of a tag definition
         case eOutsideTag:

            // Check what type of token we obtained
            switch(type)
            {
               // If we have found text or quoted text
            case eTokenText:
            case eTokenCloseTag:          /* '>'         */
            case eTokenShortHandClose:    /* '/>'        */
            case eTokenQuotedText:
            case eTokenEquals:
               break;

               // If we found a start tag '<' and declarations '<?'
            case eTokenTagStart:
            case eTokenDeclaration:

               // Cache whether this new element is a declaration or not
               nDeclaration = (type == eTokenDeclaration);

               // If we have node text then add this to the element
               if (maybeAddTxT(pXML,token.pStr)) return FALSE;

               // Find the name of the tag
               token = GetNextToken(pXML, &cbToken, &type);

               // Return an error if we couldn't obtain the next token or
               // it wasnt text
               if (type != eTokenText)
               {
                  pXML->error = eXMLErrorMissingTagName;
                  return FALSE;
               }

               // If we found a new element which is the same as this
               // element then we need to pass this back to the caller..













               // If the name of the new element differs from the name of
               // the current element we need to add the new element to
               // the current one and recurse
               pNew = addChild_priv(MEMORYINCREASE,stringDup(token.pStr,cbToken), nDeclaration,-1);

               while (!pNew.isEmpty())
               {
                  // Callself to process the new node.  If we return
                  // FALSE this means we dont have any more
                  // processing to do...

                  if (!pNew.ParseXMLElement(pXML)) return FALSE;
                  else
                  {
                     // If the call to recurse this function
                     // evented in a end tag specified in XML then
                     // we need to unwind the calls to this
                     // function until we find the appropriate node
                     // (the element name and end tag name must
                     // match)
                     if (pXML->cbEndTag)
                     {
                        // If we are back at the root node then we
                        // have an unmatched end tag
                        if (!d->lpszName)
                        {
                           pXML->error=eXMLErrorUnmatchedEndTag;
                           return FALSE;
                        }

                        // If the end tag matches the name of this
                        // element then we only need to unwind
                        // once more...

                        if (myTagCompare(d->lpszName, pXML->lpEndTag)==0)
                        {
                           pXML->cbEndTag = 0;
                        }

                        return TRUE;
                     } else
                        if (pXML->cbNewElement)
                        {
                           // If the call indicated a new element is to
                           // be created on THIS element.

                           // If the name of this element matches the
                           // name of the element we need to create
                           // then we need to return to the caller
                           // and let it process the element.

                           if (myTagCompare(d->lpszName, pXML->lpNewElement)==0)
                           {
                              return TRUE;
                           }

                           // Add the new element and recurse
                           pNew = addChild_priv(MEMORYINCREASE,stringDup(pXML->lpNewElement,pXML->cbNewElement),0,-1);
                           pXML->cbNewElement = 0;
                        }
                        else
                        {
                           // If we didn't have a new element to create
                           pNew = emptyXMLNode;

                        }
                  }
               }

               break;

               // If we found an end tag
            case eTokenTagEnd:

               // If we have node text then add this to the element
               if (maybeAddTxT(pXML,token.pStr)) return FALSE;

               // Find the name of the end tag
               token = GetNextToken(pXML, &cbTemp, &type);

               // The end tag should be text
               if (type != eTokenText)
               {
                  pXML->error = eXMLErrorMissingEndTagName;
                  return FALSE;
               }
               lpszTemp = token.pStr;

               // After the end tag we should find a closing tag
               token = GetNextToken(pXML, &cbToken, &type);
               if (type != eTokenCloseTag)
               {
                  pXML->error = eXMLErrorMissingEndTagName;
                  return FALSE;
               }
               pXML->lpszText=pXML->lpXML+pXML->nIndex;

               // We need to return to the previous caller.  If the name
               // of the tag cannot be found we need to keep returning to
               // caller until we find a match
               if (myTagCompare(d->lpszName, lpszTemp) != 0)
#ifdef STRICT_PARSING
               {
                  pXML->error=eXMLErrorUnmatchedEndTag;
                  pXML->nIndexMissigEndTag=pXML->nIndex;
                  return FALSE;
               }
#else
               {
                  pXML->error=eXMLErrorMissingEndTag;
                  pXML->nIndexMissigEndTag=pXML->nIndex;
                  pXML->lpEndTag = lpszTemp;
                  pXML->cbEndTag = cbTemp;
               }
#endif

               // Return to the caller
               exactMemory(d);
               return TRUE;

               // If we found a clear (unformatted) token
            case eTokenClear:
               // If we have node text then add this to the element
               if (maybeAddTxT(pXML,token.pStr)) return FALSE;
               if (parseClearTag(pXML, token.pClr)) return FALSE;
               pXML->lpszText=pXML->lpXML+pXML->nIndex;
               break;

            default:
               break;
            }
            break;

            // If we are inside a tag definition we need to search for attributes
         case eInsideTag:

            // Check what part of the attribute (name, equals, value) we
            // are looking for.
            switch(attrib)
            {
               // If we are looking for a new attribute
            case eAttribName:

               // Check what the current token type is
               switch(type)
               {
                  // If the current type is text...
                  // Eg.  'attribute'
               case eTokenText:
                  // Cache the token then indicate that we are next to
                  // look for the equals
                  lpszTemp = token.pStr;
                  cbTemp = cbToken;
                  attrib = eAttribEquals;
                  break;

                  // If we found a closing tag...
                  // Eg.  '>'
               case eTokenCloseTag:
                  // We are now outside the tag
                  status = eOutsideTag;
                  pXML->lpszText=pXML->lpXML+pXML->nIndex;
                  break;

                  // If we found a short hand '/>' closing tag then we can
                  // return to the caller
               case eTokenShortHandClose:
                  exactMemory(d);
                  pXML->lpszText=pXML->lpXML+pXML->nIndex;
                  return TRUE;

                  // Errors...
               case eTokenQuotedText:    /* '"SomeText"'   */
               case eTokenTagStart:      /* '<'            */
               case eTokenTagEnd:        /* '</'           */
               case eTokenEquals:        /* '='            */
               case eTokenDeclaration:   /* '<?'           */
               case eTokenClear:
                  pXML->error = eXMLErrorUnexpectedToken;
                  return FALSE;
               default: break;
               }
               break;

               // If we are looking for an equals
            case eAttribEquals:
               // Check what the current token type is
               switch(type)
               {
                  // If the current type is text...
                  // Eg.  'Attribute AnotherAttribute'
               case eTokenText:
                  // Add the unvalued attribute to the list
                  addAttribute_priv(MEMORYINCREASE,stringDup(lpszTemp,cbTemp), OSAL_NULL);
                  // Cache the token then indicate.  We are next to
                  // look for the equals attribute
                  lpszTemp = token.pStr;
                  cbTemp = cbToken;
                  break;

                  // If we found a closing tag 'Attribute >' or a short hand
                  // closing tag 'Attribute />'
               case eTokenShortHandClose:
               case eTokenCloseTag:
                  // If we are a declaration element '<?' then we need
                  // to remove extra closing '?' if it exists
                  pXML->lpszText=pXML->lpXML+pXML->nIndex;

                  if(OSAL_NULL == lpszTemp)
                  {
                     return OSAL_ERROR;
                  }
                  else
                  {
                  if (d->isDeclaration &&
                     (lpszTemp[cbTemp-1]) == _T('?'))
                  {
                     cbTemp--;
                  }
                  }

                  if (cbTemp)
                  {
                     // Add the unvalued attribute to the list
                     addAttribute_priv(MEMORYINCREASE,stringDup(lpszTemp,cbTemp), OSAL_NULL);
                  }

                  // If this is the end of the tag then return to the caller
                  if (type == eTokenShortHandClose)
                  {
                     exactMemory(d);
                     return TRUE;
                  }

                  // We are now outside the tag
                  status = eOutsideTag;
                  break;

                  // If we found the equals token...
                  // Eg.  'Attribute ='
               case eTokenEquals:
                  // Indicate that we next need to search for the value
                  // for the attribute
                  attrib = eAttribValue;
                  break;

                  // Errors...
               case eTokenQuotedText:    /* 'Attribute "InvalidAttr"'*/
               case eTokenTagStart:      /* 'Attribute <'            */
               case eTokenTagEnd:        /* 'Attribute </'           */
               case eTokenDeclaration:   /* 'Attribute <?'           */
               case eTokenClear:
                  pXML->error = eXMLErrorUnexpectedToken;
                  return FALSE;
               default: break;
               }
               break;

               // If we are looking for an attribute value
            case eAttribValue:
               // Check what the current token type is
               switch(type)
               {
                  // If the current type is text or quoted text...
                  // Eg.  'Attribute = "Value"' or 'Attribute = Value' or
                  // 'Attribute = 'Value''.
               case eTokenText:
               case eTokenQuotedText:
                  // If we are a declaration element '<?' then we need
                  // to remove extra closing '?' if it exists
                  if (d->isDeclaration &&
                     (token.pStr[cbToken-1]) == _T('?'))
                  {
                     cbToken--;
                  }

                  if (cbTemp)
                  {
                     // Add the valued attribute to the list
                     if (type==eTokenQuotedText) { token.pStr++; cbToken-=2; }
                     XMLCSTR attrVal=token.pStr;
                     if (attrVal)
                     {
                        attrVal=fromXMLString(attrVal,cbToken,pXML);
                        if (!attrVal) return FALSE;
                     }
                     addAttribute_priv(MEMORYINCREASE,stringDup(lpszTemp,cbTemp),attrVal);
                  }

                  // Indicate we are searching for a new attribute
                  attrib = eAttribName;
                  break;

                  // Errors...
               case eTokenTagStart:        /* 'Attr = <'          */
               case eTokenTagEnd:          /* 'Attr = </'         */
               case eTokenCloseTag:        /* 'Attr = >'          */
               case eTokenShortHandClose:  /* "Attr = />"         */
               case eTokenEquals:          /* 'Attr = ='          */
               case eTokenDeclaration:     /* 'Attr = <?'         */
               case eTokenClear:
                  pXML->error = eXMLErrorUnexpectedToken;
                  return FALSE;
                  break;
               default: break;
               }
            }
         }
      }
      // If we failed to obtain the next token
      else
      {
         if ((!d->isDeclaration)&&(d->pParent))
         {
#ifdef STRICT_PARSING
            pXML->error=eXMLErrorUnmatchedEndTag;
#else
            pXML->error=eXMLErrorMissingEndTag;
#endif
            pXML->nIndexMissigEndTag=pXML->nIndex;
         }
         return FALSE;
      }
   }
}

// Count the number of lines and columns in an XML string.
static tVoid CountLinesAndColumns(XMLCSTR lpXML, tS32 nUpto, XMLResults *pResults)
{
   XMLCHAR ch;
   //assert(lpXML);
   //assert(pResults);

   struct XML xml={ lpXML,lpXML, 0, 0, eXMLErrorNone, OSAL_NULL, 0, OSAL_NULL, 0, TRUE };

   pResults->nLine = 1;
   pResults->nColumn = 1;
   while (xml.nIndex<nUpto)
   {
      ch = getNextChar(&xml);
      if (ch != _T('\n')) pResults->nColumn++;
      else
      {
         pResults->nLine++;
         pResults->nColumn=1;
      }
   }
}

// Parse XML and return the root element.
XMLNode XMLNode::parseString(XMLCSTR lpszXML, XMLCSTR tag, XMLResults *pResults)
{
   if (!lpszXML)
   {
      if (pResults)
      {
         pResults->error=eXMLErrorNoElements;
         pResults->nLine=0;
         pResults->nColumn=0;
      }
      return emptyXMLNode;
   }

   XMLNode xnode(OSAL_NULL,OSAL_NULL,FALSE);
   struct XML xml={ lpszXML, lpszXML, 0, 0, eXMLErrorNone, OSAL_NULL, 0, OSAL_NULL, 0, TRUE };

   // Create header element
   xnode.ParseXMLElement(&xml);
   enum XMLError error = xml.error;
   if ((xnode.nChildNode()==1)&&(xnode.nElement()==1)) xnode=xnode.getChildNode(); // skip the empty node

   // If no error occurred
   if ((error==eXMLErrorNone)||(error==eXMLErrorMissingEndTag))
   {
      XMLCSTR name=xnode.getName();
      if (tag&&OSAL_u32StringLength(tag)&&((!name)||(OSAL_MyStricmp(xnode.getName(),tag))))
      {
         XMLNode nodeTmp;
         tS32 i=0;
         while (i<xnode.nChildNode())
         {
            nodeTmp=xnode.getChildNode(i);
            if (OSAL_MyStricmp(nodeTmp.getName(),tag)==0) break;
            if (nodeTmp.isDeclarationSet()) { xnode=nodeTmp; i=0; } else i++;
         }
         if (i>=xnode.nChildNode())
         {
            if (pResults)
            {
               pResults->error=eXMLErrorFirstTagNotFound;
               pResults->nLine=0;
               pResults->nColumn=0;
            }
            return emptyXMLNode;
         }
         xnode=nodeTmp;
      }
   } else
   {
      // Cleanup: this will destroy all the nodes
      xnode = emptyXMLNode;
   }


   // If we have been given somewhere to place results
   if (pResults)
   {
      pResults->error = error;

      // If we have an error
      if (error!=eXMLErrorNone)
      {
         if (error==eXMLErrorMissingEndTag) xml.nIndex=xml.nIndexMissigEndTag;
         // Find which line and column it starts on.
         CountLinesAndColumns(xml.lpXML, xml.nIndex, pResults);
      }
   }
   return xnode;
}


XMLNode XMLNode::parseFile(XMLCSTR filename, XMLCSTR tag, XMLResults *pResults)
{
   if (pResults) { pResults->nLine=0; pResults->nColumn=0; }

   OSAL_tIODescriptor fd =OSAL_IOOpen(filename,OSAL_EN_READONLY);
   if (fd==OSAL_ERROR) 
   { 
      if (pResults) pResults->error=eXMLErrorFileNotFound; 
      return emptyXMLNode; 
   }
   OSALUTIL_s32FSeek(fd, 0, OSAL_C_S32_SEEK_END);
   tS32 l=OSALUTIL_s32FTell(fd),headerSz=0;
   if (l <= 0)
   { 
      if (pResults) pResults->error=eXMLErrorEmpty; 
      OSAL_s32IOClose(fd);
      return emptyXMLNode; 
   }
   OSALUTIL_s32FSeek(fd, 0, OSAL_C_S32_SEEK_SET);
   tPU8 buf=(tPU8)OSAL_MyMalloc(l+1);
   OSAL_s32IORead(fd,(tPS8)buf,l*1);

   OSAL_s32IOClose(fd);
   buf[l]=0;
   if ((buf[0]==0xef)&&(buf[1]==0xbb)&&(buf[2]==0xbf)) headerSz=3;
   if (!buf) 
   { 
      if (pResults) pResults->error=eXMLErrorCharConversionError; 
      return emptyXMLNode; 
   }
   XMLNode x=parseString((XMLSTR)(buf+headerSz),tag,pResults);
   OSAL_MyFree(buf);
   return x;
}


static inline tVoid charmemset(XMLSTR dest,XMLCHAR c,tS32 l) { while (l--) *(dest++)=c; }
// private:
// Creates an user friendly XML string from a given element with
// appropriate white space and carriage returns.
//
// This recurses through all subnodes then adds contents of the nodes to the
// string.
tS32 XMLNode::CreateXMLStringR(XMLNodeData *pEntry, XMLSTR lpszMarker, tS32 nFormat)
{
   tS32 nResult = 0;
   tS32 cb;
   tS32 cbElement;
   tS32 nChildFormat=-1;
   tS32 nElementI=pEntry->nChild+pEntry->nText+pEntry->nClear;
   tS32 i,j;

   assert(pEntry);

#define LENSTR(lpsz) (lpsz ? OSAL_u32StringLength(lpsz) : 0)

   // If the element has no name then assume this is the head node.
   cbElement = (tS32)LENSTR(pEntry->lpszName);

   if (cbElement)
   {
      // "<elementname "
      cb = nFormat == -1 ? 0 : nFormat;

      if (lpszMarker)
      {
         if (cb) charmemset(lpszMarker, INDENTCHAR, sizeof(XMLCHAR)*cb);
         nResult = cb;
         lpszMarker[nResult++]=_T('<');
         if (pEntry->isDeclaration) lpszMarker[nResult++]=_T('?');
         OSAL_szStringCopy(&lpszMarker[nResult], pEntry->lpszName);
         nResult+=cbElement;
         lpszMarker[nResult++]=_T(' ');

      } else
      {
         nResult+=cbElement+2+cb;
         if (pEntry->isDeclaration) nResult++;
      }

      // Enumerate attributes and add them to the string
      XMLAttribute *pAttr=pEntry->pAttribute;
      for (i=0; i<pEntry->nAttribute; i++)
      {
         // "Attrib
         cb = (tS32)LENSTR(pAttr->lpszName);
         if (cb)
         {
            if (lpszMarker) OSAL_szStringCopy(&lpszMarker[nResult], pAttr->lpszName);
            nResult += cb;
            // "Attrib=Value "
            if (pAttr->lpszValue)
            {
               cb=(tS32)lengthXMLString(pAttr->lpszValue);
               if (lpszMarker)
               {
                  lpszMarker[nResult]=_T('=');
                  lpszMarker[nResult+1]=_T('"');
                  if (cb) toXMLString(&lpszMarker[nResult+2],pAttr->lpszValue);
                  lpszMarker[nResult+cb+2]=_T('"');
               }
               nResult+=cb+3;
            }
            if (lpszMarker) lpszMarker[nResult] = _T(' ');
            nResult++;
         }
         pAttr++;
      }

      if (pEntry->isDeclaration)
      {
         if (lpszMarker)
         {
            lpszMarker[nResult-1]=_T('?');
            lpszMarker[nResult]=_T('>');
         }
         nResult++;
         if (nFormat!=-1)
         {
            if (lpszMarker) lpszMarker[nResult]=_T('\n');
            nResult++;
         }
      } else
         // If there are child nodes we need to terminate the start tag
         if (nElementI)
         {
            if (lpszMarker) lpszMarker[nResult-1]=_T('>');
            if (nFormat!=-1)
            {
               if (lpszMarker) lpszMarker[nResult]=_T('\n');
               nResult++;
            }
         } else nResult--;
   }

   // Calculate the child format for when we recurse.  This is used to
   // determine the number of spaces used for prefixes.
   if (nFormat!=-1)
   {
      if (cbElement&&(!pEntry->isDeclaration)) nChildFormat=nFormat+1;
      else nChildFormat=nFormat;
   }

   // Enumerate through remaining children
   for (i=0; i<nElementI; i++)
   {
      j=pEntry->pOrder[i];
      switch((XMLElementType)(j&3))
      {
         // Text nodes
      case eNodeText:
         {
            // "Text"
            XMLCSTR pChild=pEntry->pText[j>>2];
            cb = (tS32)lengthXMLString(pChild);
            if (cb)
            {
               if (nFormat!=-1)
               {
                  if (lpszMarker)
                  {
                     charmemset(&lpszMarker[nResult],INDENTCHAR,sizeof(XMLCHAR)*(nFormat + 1));
                     toXMLString(&lpszMarker[nResult+nFormat+1],pChild);
                     lpszMarker[nResult+nFormat+1+cb]=_T('\n');
                  }
                  nResult+=cb+nFormat+2;
               } else
               {
                  if (lpszMarker) toXMLString(&lpszMarker[nResult], pChild);
                  nResult += cb;
               }
            }
            break;
         }

         // Clear type nodes
      case eNodeClear:
         {
            XMLClear *pChild=pEntry->pClear+(j>>2);
            // "OpenTag"
            cb = (tS32)LENSTR(pChild->lpszOpenTag);
            if (cb)
            {
               if (nFormat!=-1)
               {
                  if (lpszMarker)
                  {
                     charmemset(&lpszMarker[nResult], INDENTCHAR, sizeof(XMLCHAR)*(nFormat + 1));
                     OSAL_szStringCopy(&lpszMarker[nResult+nFormat+1], pChild->lpszOpenTag);
                  }
                  nResult+=cb+nFormat+1;
               }
               else
               {
                  if (lpszMarker)OSAL_szStringCopy(&lpszMarker[nResult], pChild->lpszOpenTag);
                  nResult += cb;
               }
            }

            // "OpenTag Value"
            cb = (tS32)LENSTR(pChild->lpszValue);
            if (cb)
            {
               if (lpszMarker) OSAL_szStringCopy(&lpszMarker[nResult], pChild->lpszValue);
               nResult += cb;
            }

            // "OpenTag Value CloseTag"
            cb = (tS32)LENSTR(pChild->lpszCloseTag);
            if (cb)
            {
               if (lpszMarker) OSAL_szStringCopy(&lpszMarker[nResult], pChild->lpszCloseTag);
               nResult += cb;
            }

            if (nFormat!=-1)
            {
               if (lpszMarker) lpszMarker[nResult] = _T('\n');
               nResult++;
            }
            break;
         }

         // Element nodes
      case eNodeChild:
         {
            // Recursively add child nodes
            nResult += CreateXMLStringR(pEntry->pChild[j>>2].d, lpszMarker ? lpszMarker + nResult : 0, nChildFormat);
            break;
         }
      default: break;
      } //lint !e788
   }

   if ((cbElement)&&(!pEntry->isDeclaration))
   {
      // If we have child entries we need to use long XML notation for
      // closing the element - "<elementname>blah blah blah</elementname>"
      if (nElementI)
      {
         // "</elementname>\0"
         if (lpszMarker)
         {
            if (nFormat != -1)
            {
               if (nFormat)
               {
                  charmemset(&lpszMarker[nResult], INDENTCHAR,sizeof(XMLCHAR)*nFormat);
                  nResult+=nFormat;
               }
            }

            OSAL_szStringCopy(&lpszMarker[nResult], _T("</"));
            nResult += 2;
            OSAL_szStringCopy(&lpszMarker[nResult], pEntry->lpszName);
            nResult += cbElement;

            if (nFormat == -1)
            {
               OSAL_szStringCopy(&lpszMarker[nResult], _T(">"));
               nResult++;
            } else
            {
               OSAL_szStringCopy(&lpszMarker[nResult], _T(">\n"));
               nResult+=2;
            }
         } else
         {
            if (nFormat != -1) nResult+=cbElement+4+nFormat;
            else nResult+=cbElement+3;
         }
      } else
      {
         // If there are no children we can use shorthand XML notation -
         // "<elementname/>"
         // "/>\0"
         if (lpszMarker)
         {
            if (nFormat == -1)
            {
               OSAL_szStringCopy(&lpszMarker[nResult], _T("/>"));
               nResult += 2;
            }
            else
            {
               OSAL_szStringCopy(&lpszMarker[nResult], _T("/>\n"));
               nResult += 3;
            }
         }
         else
         {
            nResult += nFormat == -1 ? 2 : 3;
         }
      }
   }

   return nResult;
}

#undef LENSTR

// Create an XML string
// @param       tS32 nFormat             - 0 if no formatting is required
//                                        otherwise nonzero for formatted text
//                                        with carriage returns and indentation.
// @param       tPS32 pnSize             - [out] pointer to the size of the
//                                        returned string not including the
//                                        NULL terminator.
// @return      XMLSTR                  - Allocated XML string, you must free
//                                        this with free().
XMLSTR XMLNode::createXMLString(tS32 nFormat, tPS32 pnSize) const
{
   if (!d) { if (pnSize) *pnSize=0; return OSAL_NULL; }

   XMLSTR lpszResult = OSAL_NULL;
   tS32 cbStr;

   // Recursively Calculate the size of the XML string
   if (!dropWhiteSpace) nFormat=0;
   nFormat = nFormat ? 0 : -1;
   cbStr = CreateXMLStringR(d, 0, nFormat);
   assert(cbStr);
   // Alllocate memory for the XML string + the OSAL_NULL terminator and
   // create the recursively XML string.
   lpszResult=(XMLSTR)OSAL_MyMalloc((cbStr+1)*sizeof(XMLCHAR));
   CreateXMLStringR(d, lpszResult, nFormat);
   if (pnSize) *pnSize = cbStr;
   return lpszResult;
}

/*
XMLNode::~XMLNode() 
{ 
   deleteNodeContent(); 
}
*/
XMLNode::~XMLNode()
{
   if (!d) return;
   d->ref_count--;
   emptyTheNode(0);
}


tS32 XMLNode::detachFromParent(XMLNodeData *dq)
{
   XMLNode *pa=dq->pParent->pChild;
   tS32 i=0;
   while (((tPVoid)(pa[i].d))!=((tPVoid)dq)) i++;
   dq->pParent->nChild--;
   if (dq->pParent->nChild) OSAL_pvMemoryMove(pa+i,pa+i+1,(dq->pParent->nChild-i)*sizeof(XMLNode));
   else { OSAL_MyFree(pa); dq->pParent->pChild=OSAL_NULL; }
   return removeOrderElement(dq->pParent,eNodeChild,i);
}

/*
tVoid XMLNode::deleteNodeContent(tChar force)
{
   tS32 i = 0;

   if(d)
   {
      (d->ref_count) --;
      if ((d->ref_count==0)||force)
      {
         if (d->pParent)
         {
            detachFromParent(d);
         }

         for(i=0; i<d->nChild; i++) 
         { 
            d->pChild[i].d->pParent=OSAL_NULL; 
            d->pChild[i].deleteNodeContent(force); 
         }
         OSAL_MyFree(d->pChild);

         for(i=0; i<d->nText; i++) 
         {
            OSAL_MyFree((tPVoid)d->pText[i]);
         }
         OSAL_MyFree(d->pText);

         for(i=0; i<d->nClear; i++)
         {
            OSAL_MyFree((tPVoid)d->pClear[i].lpszValue);
         }
         OSAL_MyFree(d->pClear);

         for(i=0; i<d->nAttribute; i++)
         {
            OSAL_MyFree((tPVoid)d->pAttribute[i].lpszName);
            OSAL_MyFree((tPVoid)d->pAttribute[i].lpszValue);
         }
         OSAL_MyFree(d->pAttribute);

         OSAL_MyFree(d->pOrder);
         OSAL_MyFree((tPVoid)d->lpszName);
         OSAL_MyFree(d);
         d=OSAL_NULL;
      }
   }
}
*/
void XMLNode::deleteNodeContent()
{
   if (!d) return;
   if (d->pParent) { detachFromParent(d); d->pParent=NULL; d->ref_count--; }
   emptyTheNode(1);
}

void XMLNode::emptyTheNode(char force)
{
   XMLNodeData *dd=d; // warning: must stay this way!
   if ((dd->ref_count==0)||force)
   {
      if (d->pParent) detachFromParent(d);
      int i;
      XMLNode *pc;
      for(i=0; i<dd->nChild; i++)
      {
         pc=dd->pChild+i;
         pc->d->pParent=NULL;
         pc->d->ref_count--;
         pc->emptyTheNode(force);
      }
      OSAL_MyFree(dd->pChild);
      for(i=0; i<dd->nText; i++) OSAL_MyFree((void*)dd->pText[i]);
      OSAL_MyFree(dd->pText);
      for(i=0; i<dd->nClear; i++) OSAL_MyFree((void*)dd->pClear[i].lpszValue);
      OSAL_MyFree(dd->pClear);
      for(i=0; i<dd->nAttribute; i++)
      {
         OSAL_MyFree((void*)dd->pAttribute[i].lpszName);
         if (dd->pAttribute[i].lpszValue) OSAL_MyFree((void*)dd->pAttribute[i].lpszValue);
      }
      OSAL_MyFree(dd->pAttribute);
      OSAL_MyFree(dd->pOrder);
      OSAL_MyFree((void*)dd->lpszName);
      dd->nChild=0;    dd->nText=0;    dd->nClear=0;    dd->nAttribute=0;
      dd->pChild=NULL; dd->pText=NULL; dd->pClear=NULL; dd->pAttribute=NULL;
      dd->pOrder=NULL; dd->lpszName=NULL; dd->pParent=NULL;
   }
   if (dd->ref_count==0)
   {
      OSAL_MyFree(dd);
      d=NULL;
   }
}


XMLNode XMLNode::addChild(XMLNode childNode, tS32 pos)
{
   XMLNodeData *dc=childNode.d;
   if ((!dc)||(!d)) return childNode;
   if (dc->pParent) { if ((detachFromParent(dc)<=pos)&&(dc->pParent==d)) pos--; } else dc->ref_count++;
   dc->pParent=d;
   //     tS32 nc=d->nChild;
   //     d->pChild=(XMLNode*)myRealloc(d->pChild,(nc+1),memoryIncrease,sizeof(XMLNode));
   d->pChild=(XMLNode*)addToOrder(0,&pos,d->nChild,d->pChild,sizeof(XMLNode),eNodeChild);
   d->pChild[pos].d=dc;
   d->nChild++;
   return childNode;
}

tVoid XMLNode::deleteAttribute(tS32 i)
{
   if ((!d)||(i<0)||(i>=d->nAttribute)) return;
   d->nAttribute--;
   XMLAttribute *p=d->pAttribute+i;
   OSAL_MyFree((tPVoid)p->lpszName);
   if (p->lpszValue) OSAL_MyFree((tPVoid)p->lpszValue);
   if (d->nAttribute) OSAL_pvMemoryMove(p,p+1,(d->nAttribute-i)*sizeof(XMLAttribute)); else { OSAL_MyFree(p); d->pAttribute=OSAL_NULL; }
}

tVoid XMLNode::deleteAttribute(XMLAttribute *a){ if (a) deleteAttribute(a->lpszName); }
tVoid XMLNode::deleteAttribute(XMLCSTR lpszName)
{
   tS32 j=0;
   getAttribute(lpszName,&j);
   if (j) deleteAttribute(j-1);
}

XMLAttribute *XMLNode::updateAttribute_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszNewName,tS32 i)
{
   if (!d) return OSAL_NULL;
   if (i>=d->nAttribute)
   {
      if (lpszNewName) return addAttribute_WOSD(lpszNewName,lpszNewValue);
      return OSAL_NULL;
   }
   XMLAttribute *p=d->pAttribute+i;
   if (p->lpszValue&&p->lpszValue!=lpszNewValue) OSAL_MyFree((tPVoid)p->lpszValue);
   p->lpszValue=lpszNewValue;
   if (lpszNewName&&p->lpszName!=lpszNewName) { OSAL_MyFree((tPVoid)p->lpszName); p->lpszName=lpszNewName; };
   return p;
}

XMLAttribute *XMLNode::updateAttribute_WOSD(XMLAttribute *newAttribute, XMLAttribute *oldAttribute)
{
   if (oldAttribute) return updateAttribute_WOSD(newAttribute->lpszValue,newAttribute->lpszName,oldAttribute->lpszName);
   return addAttribute_WOSD(newAttribute->lpszName,newAttribute->lpszValue);
}

XMLAttribute *XMLNode::updateAttribute_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszNewName,XMLCSTR lpszOldName)
{
   tS32 j=0;
   getAttribute(lpszOldName,&j);
   if (j) return updateAttribute_WOSD(lpszNewValue,lpszNewName,j-1);
   else
   {
      if (lpszNewName) return addAttribute_WOSD(lpszNewName,lpszNewValue);
      else             return addAttribute_WOSD(stringDup(lpszOldName),lpszNewValue);
   }
}

tS32 XMLNode::indexText(XMLCSTR lpszValue) const
{
   if (!d) return -1;
   tS32 i,l=d->nText;
   if (!lpszValue) { if (l) return 0; return -1; }
   XMLCSTR *p=d->pText;
   for (i=0; i<l; i++) if (lpszValue==p[i]) return i;
   return -1;
}

tVoid XMLNode::deleteText(tS32 i)
{
   if ((!d)||(i<0)||(i>=d->nText)) return;
   d->nText--;
   XMLCSTR *p=d->pText+i;
   OSAL_MyFree((tPVoid)*p);
   if (d->nText) OSAL_pvMemoryMove(p,p+1,(d->nText-i)*sizeof(XMLCSTR)); else { OSAL_MyFree(p); d->pText=OSAL_NULL; }
   removeOrderElement(d,eNodeText,i);
}

tVoid XMLNode::deleteText(XMLCSTR lpszValue) { deleteText(indexText(lpszValue)); }

XMLCSTR XMLNode::updateText_WOSD(XMLCSTR lpszNewValue, tS32 i)
{
   if (!d) return OSAL_NULL;
   if (i>=d->nText) return addText_WOSD(lpszNewValue);
   XMLCSTR *p=d->pText+i;
   if (*p!=lpszNewValue) { OSAL_MyFree((tPVoid)*p); *p=lpszNewValue; }
   return lpszNewValue;
}

XMLCSTR XMLNode::updateText_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue)
{
   if (!d) return OSAL_NULL;
   tS32 i=indexText(lpszOldValue);
   if (i>=0) return updateText_WOSD(lpszNewValue,i);
   return addText_WOSD(lpszNewValue);
}

tVoid XMLNode::deleteClear(tS32 i)
{
   if ((!d)||(i<0)||(i>=d->nClear)) return;
   d->nClear--;
   XMLClear *p=d->pClear+i;
   OSAL_MyFree((tPVoid)p->lpszValue);
   if (d->nClear) OSAL_pvMemoryMove(p,p+1,(d->nText-i)*sizeof(XMLClear)); else { OSAL_MyFree(p); d->pClear=OSAL_NULL; }
   removeOrderElement(d,eNodeClear,i);
}

tS32 XMLNode::indexClear(XMLCSTR lpszValue) const
{
   if (!d) return -1;
   tS32 i,l=d->nClear;
   if (!lpszValue) { if (l) return 0; return -1; }
   XMLClear *p=d->pClear;
   for (i=0; i<l; i++) if (lpszValue==p[i].lpszValue) return i;
   return -1;
}

tVoid XMLNode::deleteClear(XMLCSTR lpszValue) { deleteClear(indexClear(lpszValue)); }
tVoid XMLNode::deleteClear(XMLClear *a) { if (a) deleteClear(a->lpszValue); }

XMLClear *XMLNode::updateClear_WOSD(XMLCSTR lpszNewContent, tS32 i)
{
   if (!d) return OSAL_NULL;
   if (i>=d->nClear)
   {
      return addClear_WOSD(XMLClearTags[0].lpszOpen,lpszNewContent,XMLClearTags[0].lpszClose);
   }
   XMLClear *p=d->pClear+i;
   if (lpszNewContent!=p->lpszValue) { OSAL_MyFree((tPVoid)p->lpszValue); p->lpszValue=lpszNewContent; }
   return p;
}

XMLClear *XMLNode::updateClear_WOSD(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue)
{
   if (!d) return OSAL_NULL;
   tS32 i=indexClear(lpszOldValue);
   if (i>=0) return updateClear_WOSD(lpszNewValue,i);
   return addClear_WOSD(lpszNewValue,XMLClearTags[0].lpszOpen,XMLClearTags[0].lpszClose);
}

XMLClear *XMLNode::updateClear_WOSD(XMLClear *newP,XMLClear *oldP)
{
   if (oldP) return updateClear_WOSD(newP->lpszValue,oldP->lpszValue);
   return OSAL_NULL;
}

/*
XMLNode& XMLNode::operator=( const XMLNode& A )
{
   // shallow copy
   if (this != &A)
   {
      deleteNodeContent();
      d=A.d;
      if (d) (d->ref_count) ++ ;
   }
   return *this;
}
*/
XMLNode& XMLNode::operator=( const XMLNode& A )
{
   // shallow copy
   if (this != &A)
   {
      if (d) { d->ref_count--; emptyTheNode(0); }
      d=A.d; //lint !e1555 PQM_authorized_272 XMLNode::deleteNodeContent() takes care
      if (d) (d->ref_count) ++ ;
   }
   return *this;
}


XMLNode::XMLNode(const XMLNode &A)
{
   // shallow copy
   d=A.d;
   if (d) (d->ref_count)++ ;
}

tS32 XMLNode::nChildNode(XMLCSTR name) const
{
   if (!d) return 0;
   tS32 i,j=0,n=d->nChild;
   XMLNode *pc=d->pChild;
   for (i=0; i<n; i++)
   {
      if (OSAL_MyStricmp(pc->d->lpszName, name)==0) j++;
      pc++;
   }
   return j;
}

XMLNode XMLNode::getChildNode(XMLCSTR name, tPS32 j) const
{
   if (!d) return emptyXMLNode;
   tS32 i=0,n=d->nChild;
   if (j) i=*j;
   XMLNode *pc=d->pChild+i;
   for (; i<n; i++)
   {
      if (OSAL_MyStricmp(pc->d->lpszName, name)==0)
      {
         if (j) *j=i+1;
         return *pc;
      }
      pc++;
   }
   return emptyXMLNode;
}

XMLNode XMLNode::getChildNode(XMLCSTR name, tS32 j) const
{
   if (!d) return emptyXMLNode;
   tS32 i=0;
   while (j-->0) getChildNode(name,&i);
   return getChildNode(name,&i);
}

tS32 XMLNode::positionOfText     (tS32 i) const 
{ if(OSAL_NULL != d)
{
   if (i>=d->nText ) i=d->nText-1;  return findPosition(d,i,eNodeText );
}
else
{
   return OSAL_ERROR;
}
}
tS32 XMLNode::positionOfClear    (tS32 i) const 
{
   if(OSAL_NULL != d)
   {
      if (i>=d->nClear) i=d->nClear-1; return findPosition(d,i,eNodeClear);
   }
   else
   {
      return OSAL_ERROR;
   }
}
tS32 XMLNode::positionOfChildNode(tS32 i) const 
{
   if(OSAL_NULL != d)
   {
      if (i>=d->nChild) i=d->nChild-1; return findPosition(d,i,eNodeChild);
   }
   else
   {
      return OSAL_ERROR;
   }
}
tS32 XMLNode::positionOfText (XMLCSTR lpszValue) const { return positionOfText (indexText (lpszValue)); }
tS32 XMLNode::positionOfClear(XMLCSTR lpszValue) const { return positionOfClear(indexClear(lpszValue)); }
tS32 XMLNode::positionOfClear(XMLClear *a) const { if (a) return positionOfClear(a->lpszValue); return positionOfClear(); }
tS32 XMLNode::positionOfChildNode(XMLNode x)  const
{
   if ((!d)||(!x.d)) return -1;
   XMLNodeData *dd=x.d;
   XMLNode *pc=d->pChild;
   tS32 i=d->nChild;
   while (i--) if (pc[i].d==dd) return findPosition(d,i,eNodeChild);
   return -1;
}
tS32 XMLNode::positionOfChildNode(XMLCSTR name, tS32 count) const
{
   if (!name) return positionOfChildNode(count);
   tS32 j=0;
   do { getChildNode(name,&j); if (j<0) return -1; } while (count--);
   return findPosition(d,j-1,eNodeChild);
}

XMLNode XMLNode::getChildNodeWithAttribute(XMLCSTR name,XMLCSTR attributeName,XMLCSTR attributeValue, tPS32 k) const
{
   tS32 i=0,j;
   if (k) i=*k;
   XMLNode x;
   XMLCSTR t;
   do
   {
      x=getChildNode(name,&i);
      if (!x.isEmpty())
      {
         if (attributeValue)
         {
            j=0;
            do
            {
               t=x.getAttribute(attributeName,&j);
               if (t&&(OSAL_MyStricmp(attributeValue,t)==0)) { if (k) *k=i+1; return x; }
            } while (t);
         } else
         {
            if (x.isAttributeSet(attributeName)) { if (k) *k=i+1; return x; }
         }
      }
   } while (!x.isEmpty());
   return emptyXMLNode;
}

// Find an attribute on an node.
XMLCSTR XMLNode::getAttribute(XMLCSTR lpszAttrib, tPS32 j) const
{
   if (!d) return OSAL_NULL;
   tS32 i=0,n=d->nAttribute;
   if (j) i=*j;
   XMLAttribute *pAttr=d->pAttribute+i;
   for (; i<n; i++)
   {
      if (OSAL_MyStricmp(pAttr->lpszName, lpszAttrib)==0)
      {
         if (j) *j=i+1;
         return pAttr->lpszValue;
      }
      pAttr++;
   }
   return OSAL_NULL;
}

tChar XMLNode::isAttributeSet(XMLCSTR lpszAttrib) const
{
   if (!d) return FALSE;
   tS32 i,n=d->nAttribute;
   XMLAttribute *pAttr=d->pAttribute;
   for (i=0; i<n; i++)
   {
      if (OSAL_MyStricmp(pAttr->lpszName, lpszAttrib)==0)
      {
         return TRUE;
      }
      pAttr++;
   }
   return FALSE;
}

XMLCSTR XMLNode::getAttribute(XMLCSTR name, tS32 j) const
{
   if (!d) return OSAL_NULL;
   tS32 i=0;
   while (j-->0) getAttribute(name,&i);
   return getAttribute(name,&i);
}

XMLNodeContents XMLNode::enumContents(tS32 i) const
{
   XMLNodeContents c;
   if (!d) { c.type=eNodeNULL; return c; }
   if (i<d->nAttribute)
   {
      c.type=eNodeAttribute;
      c.attrib=d->pAttribute[i];
      return c;
   }
   i-=d->nAttribute;
   c.type=(XMLElementType)(d->pOrder[i]&3);
   i=(d->pOrder[i])>>2;
   switch (c.type)
   {
      case eNodeChild:     c.child = d->pChild[i];      break;
      case eNodeText:      c.text  = d->pText[i];       break;
      case eNodeClear:     c.clear = d->pClear[i];      break;
      default: break;
   } //lint !e788
   return c;
}

XMLCSTR XMLNode::getName() const { if (!d) return OSAL_NULL; return d->lpszName;   }
tS32 XMLNode::nText()       const { if (!d) return 0;    return d->nText;      }
tS32 XMLNode::nChildNode()  const { if (!d) return 0;    return d->nChild;     }
tS32 XMLNode::nAttribute()  const { if (!d) return 0;    return d->nAttribute; }
tS32 XMLNode::nClear()      const { if (!d) return 0;    return d->nClear;     }
tS32 XMLNode::nElement()    const { if (!d) return 0;    return d->nAttribute+d->nChild+d->nText+d->nClear; }
XMLClear     XMLNode::getClear         (tS32 i) const { if ((!d)||(i>=d->nClear    )) return emptyXMLClear;     return d->pClear[i];     }
XMLAttribute XMLNode::getAttribute     (tS32 i) const { if ((!d)||(i>=d->nAttribute)) return emptyXMLAttribute; return d->pAttribute[i]; }
XMLCSTR      XMLNode::getAttributeName (tS32 i) const { if ((!d)||(i>=d->nAttribute)) return OSAL_NULL;              return d->pAttribute[i].lpszName;  }
XMLCSTR      XMLNode::getAttributeValue(tS32 i) const { if ((!d)||(i>=d->nAttribute)) return OSAL_NULL;              return d->pAttribute[i].lpszValue; }
XMLCSTR      XMLNode::getText          (tS32 i) const { if ((!d)||(i>=d->nText     )) return OSAL_NULL;              return d->pText[i];      }
XMLNode      XMLNode::getChildNode     (tS32 i) const { if ((!d)||(i>=d->nChild    )) return emptyXMLNode;      return d->pChild[i];     }
XMLNode      XMLNode::getParentNode    (     ) const { if ((!d)||(!d->pParent     )) return emptyXMLNode;      return XMLNode(d->pParent); }
tChar         XMLNode::isDeclarationSet    (     ) const { if (!d) return 0;             return d->isDeclaration; }
tChar         XMLNode::isEmpty          (     ) const { return (d==OSAL_NULL); }

XMLNode       XMLNode::addChild(XMLCSTR lpszName, tChar isDeclaration, tS32 pos)
{ return addChild_priv(0,stringDup(lpszName),isDeclaration,pos); }
XMLNode       XMLNode::addChild_WOSD(XMLCSTR lpszName, tChar isDeclaration, tS32 pos)
{ return addChild_priv(0,lpszName,isDeclaration,pos); }
XMLAttribute *XMLNode::addAttribute(XMLCSTR lpszName, XMLCSTR lpszValue)
{ return addAttribute_priv(0,stringDup(lpszName),stringDup(lpszValue)); }
XMLAttribute *XMLNode::addAttribute_WOSD(XMLCSTR lpszName, XMLCSTR lpszValuev)
{ return addAttribute_priv(0,lpszName,lpszValuev); }
XMLCSTR       XMLNode::addText(XMLCSTR lpszValue, tS32 pos)
{ return addText_priv(0,stringDup(lpszValue),pos); }
XMLCSTR       XMLNode::addText_WOSD(XMLCSTR lpszValue, tS32 pos)
{ return addText_priv(0,lpszValue,pos); }
XMLClear     *XMLNode::addClear(XMLCSTR lpszValue, XMLCSTR lpszOpen, XMLCSTR lpszClose, tS32 pos)
{ return addClear_priv(0,stringDup(lpszValue),lpszOpen,lpszClose,pos); }
XMLClear     *XMLNode::addClear_WOSD(XMLCSTR lpszValue, XMLCSTR lpszOpen, XMLCSTR lpszClose, tS32 pos)
{ return addClear_priv(0,lpszValue,lpszOpen,lpszClose,pos); }
XMLCSTR       XMLNode::updateName(XMLCSTR lpszName)
{ return updateName_WOSD(stringDup(lpszName)); }
XMLAttribute *XMLNode::updateAttribute(XMLAttribute *newAttribute, XMLAttribute *oldAttribute)
{ return updateAttribute_WOSD(stringDup(newAttribute->lpszValue),stringDup(newAttribute->lpszName),oldAttribute->lpszName); }
XMLAttribute *XMLNode::updateAttribute(XMLCSTR lpszNewValue, XMLCSTR lpszNewName,tS32 i)
{ return updateAttribute_WOSD(stringDup(lpszNewValue),stringDup(lpszNewName),i); }
XMLAttribute *XMLNode::updateAttribute(XMLCSTR lpszNewValue, XMLCSTR lpszNewName,XMLCSTR lpszOldName)
{ return updateAttribute_WOSD(stringDup(lpszNewValue),stringDup(lpszNewName),lpszOldName); }
XMLCSTR       XMLNode::updateText(XMLCSTR lpszNewValue, tS32 i)
{ return updateText_WOSD(stringDup(lpszNewValue),i); }
XMLCSTR       XMLNode::updateText(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue)
{ return updateText_WOSD(stringDup(lpszNewValue),lpszOldValue); }
XMLClear     *XMLNode::updateClear(XMLCSTR lpszNewContent, tS32 i)
{ return updateClear_WOSD(stringDup(lpszNewContent),i); }
XMLClear     *XMLNode::updateClear(XMLCSTR lpszNewValue, XMLCSTR lpszOldValue)
{ return updateClear_WOSD(stringDup(lpszNewValue),lpszOldValue); }
XMLClear     *XMLNode::updateClear(XMLClear *newP,XMLClear *oldP)
{ return updateClear_WOSD(stringDup(newP->lpszValue),oldP->lpszValue); }

tVoid XMLNode::setGlobalOptions(tChar _guessUnicodeChars, tChar _strictUTF8Parsing, tChar _dropWhiteSpace)
{
   guessUnicodeChars=_guessUnicodeChars; dropWhiteSpace=_dropWhiteSpace; strictUTF8Parsing=_strictUTF8Parsing;
#ifndef _XMLUNICODE
   if (_strictUTF8Parsing) XML_ByteTable=XML_utf8ByteTable; else XML_ByteTable=XML_asciiByteTable;
#endif
}

/*
tChar XMLNode::guessUTF8ParsingParameterValue(void *buf,tS32 l, tChar useXMLEncodingAttribute)
{
if (l<25) return 0;
if (myIsTextUnicode(buf,l)) return 0;
tPU8 b=(tPU8)buf;
if ((b[0]==0xef)&&(b[1]==0xbb)&&(b[2]==0xbf)) return 1;

// Match utf-8 model ?
tS32 i=0;
while (i<l)
switch (XML_utf8ByteTable[b[i]])
{
case 4: i++; if ((i<l)&&(b[i]& 0xC0)!=0x80) return 0; // 10bbbbbb ?
case 3: i++; if ((i<l)&&(b[i]& 0xC0)!=0x80) return 0; // 10bbbbbb ?
case 2: i++; if ((i<l)&&(b[i]& 0xC0)!=0x80) return 0; // 10bbbbbb ?
case 1: i++; break;
case 0: i=l;
}
if (!useXMLEncodingAttribute) return 1;
// if encoding is specified and different from utf-8 than it's non-utf8
// otherwise it's utf-8
tChar bb[201];
l=mmin(l,200);
OSAL_pvMemoryCopy(bb,buf,l); // copy buf into bb to be able to do "bb[l]=0"
bb[l]=0;
b=(tPU8)OSAL_ps8StringSubString(bb,"encoding");
if (!b) return 1;
b+=8; while XML_isSPACECHAR(*b) b++; if (*b!='=') return 1;
b++;  while XML_isSPACECHAR(*b) b++; if ((*b!='\'')&&(*b!='"')) return 1;
b++;  while XML_isSPACECHAR(*b) b++; if ((_strnicmp((tString)b,"utf-8",5)==0)||
(_strnicmp((tString)b,"utf8",4)==0)) return 1;
return 0;
}

*/
#undef XML_isSPACECHAR
